//
// file      : adc.c
// author    : Ian Forse
// date      : 11-AUG-2002
// purpose   : ADC setup, reading and format conversion
// includes  : pic.h
// notes     : Updated 09-APR-2004
//
// Copyright (C) 2002 Ian Forse
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.

// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
// USA


#include <pic.h>
#include "adc.h"

// -------------------------------------------------------------------
// Initialising the ADC channels 
// Select the setting that you require as per 16F877 datasheet, DS30292
// --------------------------------------------------------------------

void adc_init (void)
{
ADCON1 = 0b10000000;	// SELECT ONLY ONE FROM BELOW.
}

// procedures for select combinations of ad inputs ( DS30292 page 122 )
// without external reference, right justified
         
// ADCON1 = 0b10001110;	// a0 adc input,a1,a2,a3,a5,e0,e1,e2 digital I/O
 
// ADCON1 = 0b10000100;	// a0,a1,a3 adc inputs,a2,a5,e0,e1,e2 digital I/O

// ADCON1 = 0b10000010;	// a0,a1,a2,a3,a4 adc inputs, a5,a6,a7 digital I/O

// ADCON1 = 0b10001001;	// a0,a1,a2,a3,a4,a5 adc inputs, a6,a7 digital I/O

// ADCON1 = 0b10000000;	// all adc inputs

 
// select combinations of ad inputs with positive reference on a3,
// right justified ( DS30292A page 121 )

// ADCON1 = 0b10000101;	// a0,a1 adc inputs,a4,a5,a6,a7 digital I/O

// ADCON1 = 0b10000011;	// a0,a1,a2,a4 adc inputs, a5,a6,a7 digital I/O

// ADCON1 = 0b10001010;	// a0,a1,a2,a4,a5 adc inputs, a6,a7 digital I/O

// ADCON1 = 0b10000001;	// all adc inputs


// select combinations of ad inputs with negative reference on a2 and positive 
// reference on a3, right justified ( DS30292A page 121 ) 


// ADCON1 = 0b10001111;	// a0 adc input, a1,a4,a5,a6,a7 digital I/O

// ADCON1 = 0b10001101;	// a0,a1 adc input, a4,a5,a6,a7 digital I/O
 
// ADCON1 = 0b10001100;	// a0,a1,a4 adc input, a5,a6,a7 digital I/O

// ADCON1 = 0b10001010;	// a0,a1,a4,a5 adc input, a6,a7 digital I/O
 
// ADCON1 = 0b10001000;	// a0,a1,a4,a5,a6,a7 adc input


// make all port_a digital IO


// ADCON1 = 0b10000111 -- all digital IO


// ----------------------------------------------------------------------------
// Reads the ADC level input on a specified ADC channel 64 times.
// Approximate time for one sample = 581uS
// Averages value to remove noise
// Calls conversion to obtain BCD
// Returns single 16bit word
// ----------------------------------------------------------------------------


unsigned int ReadADC(unsigned char ADC_Channel)
{
	float ADC_FILT;	
	unsigned int ADC_VALUE, i;
		if	(ADC_Channel == 0) ADCON0 = 0b01000001;
		else if	(ADC_Channel == 1) ADCON0 = 0b01001001;
		else if	(ADC_Channel == 2) ADCON0 = 0b01010001;
		else if	(ADC_Channel == 3) ADCON0 = 0b01011001;
		else if	(ADC_Channel == 4) ADCON0 = 0b01100001;
		else if	(ADC_Channel == 5) ADCON0 = 0b01101001;
		else if	(ADC_Channel == 6) ADCON0 = 0b01110001;
		else if	(ADC_Channel == 7) ADCON0 = 0b01111001;

// channel 0 = Adj Pot on dev board
// channel 1 = temperature sensor on dev board
// channel 2 = 

		ADC_FILT = 0;			// Clear out previous count
		for (i=0; i<64; i++)
			{
			ADIE	= 0;		// Masking the interrupt
			ADIF	= 0;		// Resetting the ADC interupt bit					
			ADRESL	= 0; 		// Resetting the ADRES value register
			ADRESH	= 0; 						 
			ADGO	= 1;
					  	// Staring the ADC process
			while(!ADIF) continue;	// Wait for conversion complete		

			ADC_VALUE = ADRESL;	// ADC registers are split 2 and 8 bit
			ADC_VALUE += (ADRESH << 8);	// Combined into a single integer
			
			ADC_FILT += ADC_VALUE;		 	
			}
			ADC_FILT = (ADC_FILT/64)+0.5;	// Average of 64 readings to reduce noise
							// +0.5 is for rounding			
//			ALTERNATIVE FILTER.  Change i to <255 in the for statement above
//			Comment out 3 lines below and replace with 2 lines below.  
//			Adjust delay in analog1.c
//
//			Software low pass filter to attenuate noise by -12dB
//			ADC_FILT = ADC_FILT + ((ADC_VALUE - ADC_FILT) * 0.0625);
//			}

			if (ADC_Channel == 0) ADC_VALUE = (unsigned int) (ADC_FILT*4.83);// Scaled to read 0 - 5V
			else ;			// All other channels un-scaled
			ADC_VALUE = bintobcd (ADC_VALUE);	// convert to BCD
			return (ADC_VALUE);		// Return the value of the ADC process 
}


// ----------------------------------------------------------------------------
// Converts binary (0x0000 to 0x03ff) to Binary coded decimal (0000 to 1023).
// Value returned as a single 16bit word.
// ----------------------------------------------------------------------------


unsigned int bintobcd (unsigned int p)
{                                          
	unsigned int w, r, i;
	i=16;   
	r=0; 
		while(1)
		{
			r<<=1;
			if (p & 0x8000) r|=1;
			p<<=1;
			if (--i ==0 ) break;
			w=0x03+r; 
			if (w & 0x08) r=w;
			w=0x30+r;
			if (w & 0x80) r=w;
			w=0x0300u+r;
			if (w & 0x0800u) r=w;	
			w=0x3000u+r;
			if (w & 0x8000u) r=w;	
		}
			return r;
}
