//
// file      : oem4.c
// author    : Ian Forse
// date      : 24-AUG-2004
// purpose   : OEM4-LED display driver
// includes  : None
// notes     : 
//
// Copyright (C) 2004 Lascar Electronics Ltd
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.

// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
// USA

// ----------------------------------------------------------------------------
// LED Display segment lookup table
// ----------------------------------------------------------------------------

#include "oem4.h"
extern unsigned char START_AND_LED;

const unsigned char table7 [] = {
//	  0GFEDCBA	    Hex
			//  Char
	0b00111111,	//   0     
	0b00000110,	//   1
	0b01011011,	//   2
	0b01001111,	//   3
	0b01100110,	//   4    
	0b01101101,	//   5
	0b01111101,	//   6
	0b00000111,	//   7
	0b01111111,	//   8
	0b01101111,	//   9
	0b01110111,	//   A
	0b01111100,	//   B
	0b00111001,	//   C
	0b01011110,	//   D
	0b01111001,	//   E
	0b01110001,	//   F
	0b00000000,	//   Space
	0b01110110,	//   H
	0b00111000,	//   L
	0b00111111,	//   O
	0b00000001,	//   top-score
	0b01000000,	//   minus
	0b00001000,	//   under-score
	0b01010000};	//   r   


// digit segments values
unsigned char oem4_segments_1, oem4_segments_2;
unsigned char oem4_segments_3, oem4_segments_4;

// current position for oem4_write()
unsigned char oem4_cursor_position;

// DATA & CLOCK use Portd 4 & 6
#define oem4_pin_data RD4
#define oem4_pin_clock RD6
#define oem4_pin_data_direction TRISD4
#define oem4_pin_clock_direction TRISD6
#define high (1)
#define low (0)

// ----------------------------------------------------------------------------
// Shift byte x out to the display as serial.  n is number of bits (1 x 4) + 
// (4 * 8).  Note range is from 0 i.e. 8 is from 0 to 7.
// ----------------------------------------------------------------------------

void oem4_output_byte( unsigned char n, unsigned char x )
{
	signed char i;
		for(i=n; i>=0; i--)
		{
			if ((x>>i)&0x01)
			{
				oem4_pin_data = high;
			}
			else
			{
				oem4_pin_data = low;
			}
			oem4_pin_clock = high;
			oem4_pin_clock = low;
		}
}


// ----------------------------------------------------------------------------
// Update the display from the segment values.
// ----------------------------------------------------------------------------

void oem4_update_from_segments(void)
{
	oem4_output_byte( 3, (START_AND_LED | 0x08) );
	oem4_output_byte( 7, oem4_segments_4 );
	oem4_output_byte( 7, oem4_segments_3 );
	oem4_output_byte( 7, oem4_segments_2 );
	oem4_output_byte( 7, oem4_segments_1 );
}


// ----------------------------------------------------------------------------
// Adjust segment values to either all on or all off.  Could be dropped if code
// space was tight.
// ----------------------------------------------------------------------------

void oem4_all( unsigned char x)
{
	unsigned char v;
	if (x)
	{
		v = 0b11111111;
	}
	else
	{
		v = 0b00000000;
	}
	oem4_segments_1 = v;
	oem4_segments_2 = v;
	oem4_segments_3 = v;
	oem4_segments_4 = v;
	START_AND_LED = v;
	oem4_update_from_segments();
}   


// ----------------------------------------------------------------------------
// write ascii character c to position p.
// ----------------------------------------------------------------------------
 
void oem4_write_p ( unsigned char p, unsigned char c )
{
	unsigned char segments, a;
	if (p == 1)
	{
		segments = oem4_segments_1;
	}  
	else if (p == 2)
	{
		segments = oem4_segments_2;
	}   
	else if (p == 3)
	{
		segments = oem4_segments_3;
	}   
	else if (p == 4)
	{
		segments = oem4_segments_4;
	}   

	if (c == ',')			// look for ',' == clear dots
	{
		segments = segments & ! 0b10000000;
	}  
	else if (c == '.')			// look for decimal point
	{
		segments = segments | 0b10000000;
	}  
	else
	{        
		a =  table7[c];		// lookup channel a
		oem4_cursor_position = oem4_cursor_position + 1;
	}  
	segments = ( segments & 0b10000000 ) | a;	// don't overwrite the DP's

	if (p == 1)
	{ 
		oem4_segments_1 = segments;
	}  
	else if (p == 2)
	{ 
		oem4_segments_2 = segments;
	}   
	else if (p == 3)
	{
		oem4_segments_3 = segments;
	}   
	else if (p == 4)
	{ 
		oem4_segments_4 = segments;
	}   
	oem4_update_from_segments();
}


// ----------------------------------------------------------------------------
// clear the display, set cursor position to 1
// ----------------------------------------------------------------------------

void oem4_clear(void)
{
	oem4_all( 0 );   
	oem4_cursor_position = 1;
}


// ----------------------------------------------------------------------------
// write acii character c to cursor position, reset by mdm_clear.  Allows TTY
// style message to move scroll
// ----------------------------------------------------------------------------

void oem4_write(  unsigned char c )
{        
	if (oem4_cursor_position > 4)
	{
		oem4_segments_1 = oem4_segments_2;
		oem4_segments_2 = oem4_segments_3;
		oem4_segments_3 = oem4_segments_4;
		oem4_cursor_position = 4;
	}
	oem4_write_p( oem4_cursor_position, c );
}


// ----------------------------------------------------------------------------
// initialise the pin directions
// ----------------------------------------------------------------------------

void oem4_init(void)
{
	oem4_pin_data = low;
	oem4_pin_clock = low;
	oem4_pin_data_direction = low;
	oem4_pin_clock_direction = low;
}


